' HexParty Game for CMM2
' Rev 4.0.0 William M Leue 10-May-2024. Mouse Version
' Based loosely on a Google educational game

option default integer
option base 1
option angle degrees

' Constants
const DEBUG    = 1

' Sizes, locations, and Limits
const NSIZES   = 3
const CSIZE    = 20
const RSIZE    = 6
const WIDTH    = 2*CSIZE
const HEIGHT   = sqr(3)*CSIZE-1
const RINCIRC  = 0.5*CSIZE*sqr(3)
const HSPACE   = 1.5*WIDTH
const MAXCHPR  = 19
const MAXCROW  = 19
const MAXHROW  = 19
const MAXHHPR  = 13
const CX       = mm.hres\2
const CY       = mm.vres\2 - 70
const INVALID  = 999

' Puzzle sizes
const PSMALL   = 1
const PMEDIUM  = 2
const PLARGE   = 3

' Colors
const EC_MOVING = rgb(black)
const EC_FIXED  = rgb(white)
const FC_EMPTY  = rgb(gray)
const EC_DONE   = rgb(black)

' Hex Label Types
const LB_NONE   = 0
const LB_COORDS = 1
const LB_SEGNUM = 2

' Outside segment boxes
const MAX_NPART = 13
const MAX_PSEG_HEXES = 40
const PWIDTH    = 140
const PHEIGHT   = 112
const PLYGAP    = 5

' Keyboard commands
const UP        = 128
const DOWN      = 129
const LEFT      = 130
const RIGHT     = 131
const HOME      = 134
const DELETE    = 127
const BACKSPACE = 8
const ENTER     = 13
const HINT      = asc("H")
const ESC       = 27

' Text Locations
const UMSG_Y = 560
const UMSG_W = 300
const UMSG_H = 25
const BANN_Y = 1
const BANN_W = 400
const BANN_H = 50

' puzzle choice params
const NPPL         = 20
const MAX_LINES_SM = 2
const MAX_LINES_LG = 4
const MAX_NPUZZ_SM = MAX_LINES_SM*NPPL
const MAX_NPUZZ_LG = MAX_LINES_LG*NPPL
const pdir$        = "./PUZZLES/"
const SIZE_YSTEP   = 140

' Mouse
const MCHAN  = 2

' Globals
dim size   = 0
dim ncrows(NSIZES)
dim nchpr(NSIZES, MAXCHPR)
dim nhrows(NSIZES)
dim nhhpr(NSIZES, MAXHHPR)
dim bcoords(MAXCROW, MAXCHPR, 2)
dim fwd_trtable(NSIZES, MAXCROW, MAXCHPR, 2)
dim inv_trtable(NSIZES, MAXHROW, MAXHHPR, 2)
dim board(2, 2)
dim pboard(2, 2)
dim sizenames$(NSIZES) = ("Small", "Medium", "Large")
dim npart  = 0
dim pcentroids(MAX_NPART, 2)
dim seg_nhexes(MAX_NPART)
dim seg_hex_coords(MAX_NPART, MAX_PSEG_HEXES, 2)
dim seg_pixel_coords(MAX_NPART, MAX_PSEG_HEXES, 2)
dim small_seg_hex_coords(MAX_NPART, MAX_PSEG_HEXES, 2)
dim small_seg_pixel_coords(MAX_NPART, MAX_PSEG_HEXES, 2)
dim seg_diff_hex_coords(MAX_NPART, MAX_PSEG_HEXES, 2)
dim seglocindex(MAX_NPART)
dim seg_center_index(MAX_NPART)
dim seg_bb(MAX_NPART, 6)
dim plocs(MAX_NPART, 4)
dim puzz_number = 0
dim maxnp(NSIZES)
dim pallette(MAX_NPART)
dim running = 0
dim grid_center_coords(NSIZES, 2)
dim active_segment = 0
dim active_segment_box = 0
dim lclick = 0
dim lbusy = 0
dim mousex = 0
dim mousey = 0
dim won = 0
dim reset_solved = 0

' puzzle choice globals
dim puzzles(NSIZES, MAX_NPUZZ_LG)
dim solved(NSIZES, MAX_NPUZZ_LG)
dim npuzzles(NSIZES)
dim nlines(NSIZES)
dim nlast(NSIZES)
dim npl(NSIZES, MAX_LINES_LG)
dim pnstart(NSIZES, MAX_LINES_LG)
dim sdirs$(NSIZES) = ("SMALL", "MEDIUM", "LARGE")
dim chsize = PSMALL
dim chline = 1
dim chpuzzle = 1

' Main Program
'open "debug.txt" for output as #1
TitlePage
ReadHexData
ReadPallette
MaybeResetSolved
InitMouse
do
  MakePartitionSlots
  ChoosePuzzle
  DrawScreen
  HandleEvents
loop
end

' show the title page
' (Yes, the image is AI, sorry :-))
sub TitlePage
  cls
  load png "HappyHexagon.png", 100, 0
  box 500, 550, 190, 40,, rgb(black), rgb(black)
  load png "HexPartyTitle.png", 200, 530
  pause 1500
  box 280, 200, 240, 100,, rgb(black), rgb(black)
  print @(290, 210) "Want Directions? (Y,N): ";
  input "", w$  
  if LEFT$(UCASE$(w$), 1) = "Y" then ShowHelp
end sub

' Read the Hex board data
sub ReadHexData
  local s, i, j, n
  ' get number of cartesian rows and number of hexes per row
  for s = 1 to NSIZES
    read ncrows(s)
    n = 0
    for i = 1 to ncrows(s)
      read nchpr(s, i)
    next i
  next s
  ' Get the max number of partition segments per size
  for s = 1 to NSIZES
    read maxnp(s)
  next s
  for s = 1 to NSIZES
    for i = 1 to 2
      read grid_center_coords(s, i)
    next i
  next s
end sub

' Read the color pallette data and map colors
sub ReadPallette
  local i
  for i = 1 to MAX_NPART
    read pallette(i)
    map(i) = pallette(i)
  next i
  map set
end sub

' If the program is started with the command-line argument 'RESET',
' then the 'solved' flag will be set to zero for any puzzles worked on.
sub MaybeResetSolved
  if mm.cmdline$ = "RESET" then reset_solved = 1
end sub

' Initialize Mouse and Cursor
sub InitMouse
  on error skip 1
  controller mouse open MCHAN, LeftClick
  if mm.errno <> 0 then
    print "Open Mouse Error: ";mm.errmsg$
    end
  end if
  gui cursor on 1
  settick 20, UpdateCursor
end sub

' Mouse Left-Click ISR
sub LeftClick
  if not lbusy then
    lclick = 1
    mousex = mouse(X)
    mousey = mouse(Y)
  end if
end sub

' Make cursor track mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Pick the puzzle to work on
sub ChoosePuzzle
  cls
  CountPuzzles
  SortPuzzles
  ShowPuzzles
  HandleChoiceEvents
end sub

' Find all the existing puzzles and make arrays of their numbers and
' solved status.
sub CountPuzzles
  local i, n, f$, fl, r$, fpath$, buf$, s, mp
  mp = MAX_NPUZZ_SM
  for i = 1 to NSIZES
    if i = PLARGE then mp = MAX_NPUZZ_LG
    npuzzles(i) = 0
    path$ = pdir$ + sdirs$(i) + "/*.prt"
    f$ = DIR$(path$, FILE)
    do while f$ <> ""
      fl = len(f$)
      r$ = LEFT$(f$, fl-4)
      r$ = MID$(r$, 5)
      n = val(r$)
      inc npuzzles(i)
      if npuzzles(i) > mp then
        print "ERROR -- Too many puzzles in dir ";path$+sdirs$(i)
        end
      end if
      puzzles(i, npuzzles(i)) = n
      fpath$ = pdir$ + sdirs$(i) + "/" + f$
      on error skip 1
      open fpath$ for input as #2
      if mm.errno <> 0 then
        print "ERROR -- cannot open puzzle file '";fpath$;"'"
        end
      end if
      line input #2, buf$
      line input #2, buf$
      line input #2, buf$
      s = val(buf$)
      solved(i, npuzzles(i)) = s
      close #2
      f$ = DIR$()
    loop
  next i
end sub

' Change puzzle numbers and their corresponding solved flags from
' hash table order to ascending by puzzle number.
sub SortPuzzles
  local p(MAX_NPUZZ_LG), s(MAX_NPUZZ_LG), index(MAX_NPUZZ_LG), i, j
  for i = 1 to NSIZES
    math slice puzzles(), i,, p()
    math slice solved(), i,, s()
    sort p(), index(),,, npuzzles(i)
    math insert puzzles(), i,, p()
    for j = 1 to npuzzles(i)
      solved(i, j) = s(index(j))
    next j
  next i
end sub

' Show all available puzzles for all 3 sizes and let the user
' select one to work on.
sub ShowPuzzles
  local i, j, k, m$, buf$, buf2$, y, n, sp$
  cls
  text mm.hres\2, 0, "Available Puzzles", "CT", 5,, rgb(green)
  m$ = "Select: " + chr$(146) + " " + chr$(147) + " " + chr$(148) + " " + chr$(149) + " Keys"
  text 0, 8, m$, "LT",,, rgb(green)
  m$ = "Pick: ENTER Key"
  text mm.hres-1, 8, m$, "RT",,, rgb(green)
  for i = 1 to NSIZES
    m$ = sdirs$(i) + " PUZZLES:"
    y = 40+(i-1)*SIZE_YSTEP + 15
    text 0, y, m$, "LT", 4,, rgb(100, 100, 255)
    nlines(i) = (npuzzles(i)-1)\NPPL + 1
    nlast(i) = npuzzles(i) - (nlines(i)-1)*NPPL
    n = 0
    pnstart(i, 1) = 1
    for j = 1 to nlines(i)
      npl(i, j) = NPPL
      if j = nlines(i) then npl(i, j) = nlast(i)
      buf$ = "Puzzle: "
      text 0, y+15, buf$
      buf2$ = "Solved: "
      text 0, y+30, buf2$
      x = 63
      for k = 1 to npl(i, j)
        inc n
        buf$ = str$(puzzles(i, n))
        text x, y+15, buf$
        buf2$ = str$(solved(i, n))
        text x, y+30, buf2$
        inc x, 36
      next k
      if j > 1 then
        pnstart(i, j) = npl(i, j-1) + pnstart(i, j-1)
      end if
      inc y, 40
    next j
  next i
end sub

' Handle keyboard inputs for choosing a puzzle
sub HandleChoiceEvents
  local z$, cmd, mflag
  HiliteChoice chsize, chline, chpuzzle
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    mflag = 1
    select case cmd
      case UP
        if chline > 1 then
          inc chline, -1  
        else
          if chsize > 1 then
            inc chsize, -1
            chline = nlines(chsize)
          end if
        end if
        if chpuzzle > npl(chsize, chline) then chpuzzle = npl(chsize, chline)
      case DOWN
        if chline < nlines(chsize) then
          inc chline
        else
          if chsize < NSIZES then
            inc chsize
            chline = 1
          end if
        end if      
        if chpuzzle > npl(chsize, chline) then chpuzzle = npl(chsize, chline)
      case LEFT
        if chpuzzle > 1 then
          inc chpuzzle, -1
        else
          chpuzzle = npl(chsize, chline)
        end if
      case RIGHT
        if chpuzzle < npl(chsize, chline) then
          inc chpuzzle
        else
          chpuzzle = 1
        end if
      case ENTER
        mflag = 0
        size = chsize
        puzz_num = puzzles(size, chpuzzle+pnstart(chsize, chline)-1)
        InitBoard size
        LoadPartition
        exit do
      case ESC
        mflag = 0
        Quit
    end select
    if mflag then HiliteChoice chsize, chline, chpuzzle
  loop
end sub

' Hilight the current puzzle choice
sub HiliteChoice csize, cline, cpuzzle
  local x, y, w, pn
  static prev_csize = 0
  static prev_cline = 0
  static prev_cpuzzle = 0
  const YOFF = 69
  const YLINC = 40
  const YSINC = 70
  const XOFF = 63
  const XPINC = 36
  const BWIDTH = 18
  const BHEIGHT = 14
  if (prev_csize > 0) or (prev_cline > 0) or (prev_cpuzzle > 0) then
    pn = puzzles(prev_csize, prev_cpuzzle)
    y = YOFF + (prev_csize-1)*SIZE_YSTEP
    inc y, (prev_cline-1)*YLINC
    x = XOFF + (prev_cpuzzle-1)*XPINC
    box x, y, BWIDTH, BHEIGHT,, rgb(black)
  end if
  y = YOFF +(csize-1)*SIZE_YSTEP
  inc y, (cline-1)*YLINC
  pn = puzzles(csize, cpuzzle+pnstart(csize, cline)-1)
  x = XOFF + (cpuzzle-1)*XPINC
  box x, y, BWIDTH, BHEIGHT,, rgb(yellow)
  prev_csize = csize
  prev_cline = cline
  prev_cpuzzle = cpuzzle
end sub

' Initialize the Board
sub InitBoard size
  local row, h
  erase board()
  erase pboard()
  MakeBoardCoords
  dim board(ncrows(size), MAXCHPR)
  dim pboard(ncrows(size), MAXCHPR)
  for row = 1 to ncrows(size)
    for h = 1 to MAXCHPR
      board(row, h) = 0
      pboard(row, h) = 0
    next h
  next row
  running = 1
end sub
  
' Routine to Translate Symmetric Coordinates to Cartesian
sub S2C size, srow, scol, crow, ccol
  crow = srow
  ccol = (scol+nchpr(size, srow))\2 + 1
end sub

' Routine to Translate Cartesian Coordinates to Symmetric
sub C2S size, crow, ccol, srow, scol
  srow = crow
  scol = 2*ccol-nchpr(size, crow)-1
end sub

' GetSymmmetric Column Limits per Row
' dir is LEFT or RIGHT
function GetSCLim(size, srow, dir)
  local ccol
  local nc = nchpr(size, srow)
  if dir = LEFT then
    ccol = 1
  else
    ccol = nc
  end if
  GetSCLim = 2*ccol-nc
end function

' Draw the entire screen
sub DrawScreen
  local mm$
  cls
  DrawBoard
  DrawPartitionSegments 0
  m$ = "H = Hint    HOME = New Puzzle    ESCAPE = Quit    Puzzle Number: " + str$(puzz_num)
  select case size
    case PSMALL  : cat m$, "S"
    case PMEDIUM : cat m$, "M"
    case PLARGE  : cat m$, "L"
  end select
  text mm.hres\2, mm.vres-1, m$, "CB"
end sub

' Draw the board
sub DrawBoard
  local ec
  if won then
    ec = EC_DONE
  else
    ec = EC_FIXED
  end if
  DrawGrid size, CX, CY, CSIZE, 0, board(), ec, LB_NONE
end sub

' Draw the hex grid
' arguments:
'  size: grid size (1-3), cell_size = size of a hex cell
'  gx, gy: pixel location of center of grid
'  ps: if zero, draw the entire grid.
'      if non-zero only draw cells with that value.
'  grid(): the grid array to use
sub DrawGrid size, gx, gy, cell_size, ps, grid(), ec, label
  local row, h, x, y, n, p
  local r, g, b
  gui cursor hide
  for row = 1 to ncrows(size)
    n = nchpr(size, row)
    for h = 1 to n
      p = grid(row, h)
      if (ps > 0) and (ps <> p) then continue for
      if p > 0 then
        fc = map(p)
      else
        fc = FC_EMPTY
      end if
      DrawCell row, h, gx, gy, cell_size, ec, fc, label
    next h
  next row
  gui cursor show
end sub

' Draw a hexagonal cell of the grid
' arguments:
' row, h: hexgonal coordinates of the cell in the grid
' gx, gy: pixel coordinates of the center of the grid
' sz: hex size
' ec: hex edge color code
' fc: hex fill color code
' label = 0: nothing
' label = 1: draw coordinates
' label = 2: black border
sub DrawCell row, h, gx, gy, sz, ec, fc, label
  local xv(6), yv(6), n, x, y, p, ht, hs
  local m$
  ht = sqr(3)*sz - 1
  hs = 3.0*sz
  n = nchpr(size, row)
  x = int(gx - (n-1)*0.5*hs + (h-1)*hs)
  y = int(gy - (ncrows(size)\2)*0.5*ht + (row-1)*0.5*ht)
  xv(1) = x - sz            : yv(1) = y
  xv(2) = x + sz*cos(120)   : yv(2) = y - sz*sin(120)
  xv(3) = x + sz*cos(60)    : yv(3) = y - sz*sin(60)
  xv(4) = x + sz            : yv(4) = y
  xv(5) = x + sz*cos(300)   : yv(5) = y - sz*sin(300)
  xv(6) = x + sz*cos(240)   : yv(6) = y - sz*sin(240)
  polygon 6, xv(), yv(), ec, fc
  p = board(row, h)
  LabelCell row, h, x, y, p, label
end sub

' Add a label to the hex (or none)
' For coordinate labels:
'   The top pair of coordinates is the Cartesian (crow, ccol)
'   The bottom pair is the Symmetric (srow, scol)
' For Segment labels, the label is the segment character
sub LabelCell crow, ccol, x, y, p, label
  local m$, srow, scol, flag, n, n2
 select case label
    case LB_NONE
      exit sub
    case LB_COORDS
      m$ = str$(crow) + "," + str$(ccol)
      text x, y-8, m$, "CM", 7,, rgb(red), -1
      n = nchpr(size, crow)
      n2 = n\2
      C2S size, crow, ccol, srow, scol
      m$ = str$(srow) + "," + str$(scol)
      text x, y+8, m$, "CM", 7,, rgb(blue), -1
    case LB_SEGNUM
      if p > 0 then
        m$ = str$(p)
        text x, y, m$, "CM", 7,, rgb(black), -1
      end if
  end select
end sub

' Handle User events during game play
sub HandleEvents
  local z$, cmd
  static prev_mousex = 0
  static prev_mousey = 0
  z$ = INKEY$
  do
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case HOME
          exit do
        case HINT
          ShowHint
        case DELETE, BACKSPACE
          CancelActiveSegment
        case ESC
          Quit
      end select
    end if
    if lclick then
      lbusy = 1
      HandleLeftClick
      lclick = 0
      lbusy = 0
    end if
    if active_segment > 0 then
      mousex = mouse(X) : mousey = mouse(Y)
      if (mousex <> prev_mousex) or (mousey <> prev_mousey) then
        DrawSegmentSprite active_segment, mousex, mousey
        prev_mousex = mousex : prev_mousey = mousey
      end if
    end if
  loop
end sub

' Handle mouse left click
' If click in segment box and box is currently holding a segment,
'   create the sprite and make the segment active.
' If click in main grid and cell is currently part of a placed segment,
'   remove the placed segment and make it active again.
' If neither of these is true, cancel the active segment.
sub HandleLeftClick
  local p, flag
  p = GetSegBox(mousex, mousey)
  if p > 0 then
    if active_segment then
      sprite close #1
      active_segment = 0
      DrawBoard
    else
      active_segment = p
      CreateActiveSprite p
    end if
    exit sub
  end if
  p = GetSelectedSegment(mousex, mousey)
  if p >= 0 then
    if active_segment > 0 then
      DropActiveSegment active_segment, mousex, mousey
    else
      if p > 0 then
        PickUpSegment p
      end if
    end if
  else
    CancelActiveSegment
  end if  
end sub

' Create a sprite for the active segment. Draw the segment into an
' offscreen page, read the sprite from there.
sub CreateActiveSprite p
  local x, y, w, h, pxi, pyi, c, sci, pcx, pcy, gcx, gcy
  page write 1
  cls
  DrawGrid size, CX, CY, CSIZE, p, pboard(), rgb(NOTBLACK), LB_NONE
  x = seg_bb(p, 1) : y = seg_bb(p, 2)
  w = seg_bb(p, 3) - x
  h = seg_bb(p, 4) - y
  sci = seg_center_index(p)
  pcx = seg_pixel_coords(p, sci, 1)
  pcy = seg_pixel_coords(p, sci, 2)
  seg_bb(p, 5) = seg_bb(p, 1) - (pcx + CX)
  seg_bb(p, 6) = seg_bb(p, 2) - (pcy + CY)
  sprite read #1, x, y, w, h, 1
  page write 0
end sub

' Draw the current active segment sprite around the mouse position
' It does flicker a little when moved but IMHO not too badly.
sub DrawSegmentSprite p, x, y
  local sx, sy
  sx = x + seg_bb(p, 5)
  sy = y + seg_bb(p, 6)
  sprite show #1, sx, sy, 2
end sub

' Function to return the index of the segment box clicked
function GetSegBox(x,y)
  local i, px, py
  for i = 1 to MAX_NPART
    px = plocs(i, 1) : py = plocs(i, 2)
    if (x >= px) and (x <= px+PWIDTH) then
      if (y >= py) and (y <= py+PHEIGHT) then
        if plocs(i, 4) > 0 and plocs(i, 3) > 0 then
          GetSegBox = plocs(i, 4)
          active_segment_box = i
          exit function
        end if
      endif
    end if
  next i
  GetSegBox = 0
end function

' Function to return the segment index clicked on the main grid, or zero if there is
' no segment, or -1 if the click is not on a grid cell at all.
function GetSelectedSegment(x, y)
  local v, row, h, p
  GetSelectedSegment = 0
  v = GetCurrentCell(x, y)
  if v > 0 then
    row = v\100 : h = v-100*row
    p = board(row, h)
    if p > 0 then   
      GetSelectedSegment = p
      exit function
    end if
  else
    GetSelectedSegment = -1
  end if
end function

' Function returns main grid cell in cartesian coords closest to
' mouse coordinates, or zero if mouse is not in the main grid.
' Coords are encoded as (100*row+col)
function GetCurrentCell(x, y)
  local row, h, dx, dy
  local float d
  GetCurrentCell = 0
  for row = 1 to ncrows(size)
    for h = 1 to nchpr(size, row)
      dx = x - bcoords(row, h, 1)
      dy = y - bcoords(row, h, 2)
      d = sqr(dx*dx + dy*dy)
      if d <= RINCIRC then
        GetCurrentCell = 100*row+h
        exit function
      end if
    next h
  next row
end function

' Drop the current segment into the main grid from the current sprite position,
' quantized to the nearest grid hex. The drop cannot happen if any of the segment
' hexes extend outside the main grid, or if the position of the segment would overlap
' any of the already placed segments.
' The differential placement must be done in symmetric coordinates, then converted
' to cartesian coordinates.
sub DropActiveSegment p, x, y
  local cell, srow, sh, cmr, cmh, smr, smh, crow, ch, i
  local segcoords(MAXHHPR, 2)
  cell = GetCurrentCell(x, y)
  cmr = cell\100
  cmh = cell-cmr*100
  if (cmr < 1) or (cmr > ncrows(size)) then exit sub
  if (cmh < 1) or (cmh > nchpr(size, cmr)) then exit sub
  C2S size, cmr, cmh, smr, smh
  for i = 1 to seg_nhexes(p)
    k = i
    srow = seg_diff_hex_coords(p, i, 1) + smr
    sh   = seg_diff_hex_coords(p, i, 2) + smh
    S2C size, srow, sh, crow, ch
    if (crow < 1) or (crow > ncrows(size)) then
      Beep
      exit sub
    end if
    if (ch < 1) or (ch > nchpr(size, crow)) then
      Beep
      exit sub
    end if
    if board(crow, ch) > 0 then
      Beep
      exit sub
    end if
    segcoords(i, 1) = crow : segcoords(i, 2) = ch
  next i
  for i = 1 to seg_nhexes(p)
    crow = segcoords(i, 1) : ch = segcoords(i, 2)
    board(crow, ch) = p
  next i
  active_segment = 0
  sprite close #1
  DrawBoard    
  EraseSegmentFromBox active_segment_box
  active_segment_box = 0
  if IsComplete() then
    running = 0
    ShowComplete
  end if
end sub

' Pick the segment off the main grid and turn it into a movable sprite.
' (Opposite of DropSegment)
sub PickUpSegment p
  EraseSegment p
  DrawBoard
  CreateActiveSprite p
  DrawSegmentInBox p
  active_segment_box = GetSegmentBoxIndex(p)
  active_segment = p
end sub

' Erase the specified segment from the main grid
sub EraseSegment p
  local row, h
  for row = 1 to ncrows(size)
    for h = 1 to nchpr(size, row)
      if board(row, h) = p then board(row, h) = 0
    next h
  next row
end sub

' Cancel the current active segment
sub CancelActiveSegment
  on error skip 1
  sprite close #1
  active_segment = 0            
end sub

' Beep when an attempt to drop a segment fails due to an overlap
sub Beep
  play tone 800, 800, 200
end sub

' Load Partition from file
sub LoadPartition
  local p$, buf$, ps, row, h, solved
  on error skip 1
  p$ = "./PUZZLES/" + UCASE$(sizenames$(size)) + "/part" + str$(puzz_num) + ".prt"
  open p$ for input as #2
  if mm.errno <> 0 then
    cls
    print "Error opening file '";p$;"' for input: ";mm.errmsg$
    end
  end if
  line input #2, buf$
  ps = val(buf$)
  if ps <> size then
    cls
    print "Error : partition grid size does not match selected size"
    end
  end if
  line input #2, buf$
  npart = val(buf$)
  line input #2, buf$
  solved = val(buf$)
  for row = 1 to ncrows(size)
    line input #2, buf$
    for h = 1 to nchpr(size, row)
      pboard(row, h) = val(field$(buf$, h, ","))
    next h
  next row
  close #2
  DrawGrid size, CX, CY, CSIZE, 0, board(), LB_SEGNUM
  MakeSegmentPositions CSIZE, seg_hex_coords(), seg_pixel_coords()
  GetSegCenters
  MakeSegBBs
  AssignPartitionSegments
  MakeSegmentPositions RSIZE, small_seg_hex_coords(), small_seg_pixel_coords()
  MakeCentroids RSIZE
  DrawPartitionSegments 1
end sub

' Save the Partition to disk
' This is only done when a puzzle has been solved. It updates the 'solved'
' value in the puzzle to 1.
sub SavePartition solved
  local np, n, path$, i, row, h
  path$ = "./PUZZLES/" + sizenames$(size) + "/part" + str$(puzz_num) + ".prt"
  on error skip 1
  open path$ for output as #2
  if mm.errno <> 0 then
    cls
    print "Error opening '";path$;" for output: ";mm.errmsg$
    end
  end if
  print #2, str$(size)
  print #2, str$(npart)
  print #2, str$(solved)
  for row = 1 to ncrows(size)
    n = nchpr(size, row)
    for h = 1 to n
      print #2, str$(board(row, h)) + ",";
    next h
    print #2, ""
  next row
  close #2
end sub

' Make the permanent board cell coordinates
sub MakeBoardCoords
  local row, h, n
  local float ht, hs
  for row = 1 to ncrows(size)
    n = nchpr(size, row)
    for h = 1 to n
      ht = sqr(3)*CSIZE - 1
      hs = 3.0*CSIZE
      x = int(CX - (n-1)*0.5*hs + (h-1)*hs)
      y = int(CY - (ncrows(size)\2)*0.5*ht + (row-1)*0.5*ht)
      bcoords(row, h, 1) = x
      bcoords(row, h, 2) = y
    next h
  next row
end sub

' Make the segment position arrays
' arguments:
'   cell_size: the hex cell size (CSIZE for main grid, RSIZE for segment boxes)
'   hcoords: the array of hex coordinates to fill
'   pcoords: the array of pixel coordinates to fill
' This is used for both the main grid and the segment boxes, but with different
' argument values.
sub MakeSegmentPositions cell_size, hcoords(), pcoords()
  local row, h, i, n, p
  local index(MAX_PSEG_HEXES)
  local float x, y
  local float ht = sqr(3)*cell_size - 1
  local float hs = 3.0*cell_size
  for i = 1 to npart
    index(i) = 0
    seg_nhexes(i) = 0
  next i
  for row = 1 to ncrows(size)
    n = nchpr(size, row)
    y = -(ncrows(size)\2)*0.5*ht + (row-1)*0.5*ht
    for h = 1 to n
      x = -(n-1)*0.5*hs + (h-1)*hs
      p = pboard(row, h)
      if p = 0 then
        cls
        print "ERROR -- empty cell at (";row;",";h;")"
        end
      end if
      inc index(p)
      inc seg_nhexes(p)
      hcoords(p, index(p), 1) = row
      hcoords(p, index(p), 2) = h
      pcoords(p, index(p), 1) = x
      pcoords(p, index(p), 2) = y
    next h
  next row
end sub

' Make the slots for partition segments up to 13 for a large puzzle
sub MakePartitionSlots
  local i, j, x, y, n
  n = 0
  x = 5 : y = 1
  for i = 1 to 5
    inc n
    plocs(n, 1) = x : plocs(n, 2) = y : plocs(n, 3) = -1
    box x, y, PWIDTH, PHEIGHT,, rgb(gray)
    inc y, PHEIGHT+PLYGAP
  next i
  x = 650 : y = 1
  for i = 1 to 5
    inc n
    plocs(n, 1) = x : plocs(n, 2) = y : plocs(n, 3) = -1
    box x, y, PWIDTH, PHEIGHT,, rgb(gray)
    inc y, PHEIGHT+PLYGAP
  next i
  x = 168 : y = 461
  for i = 1 to 3
    inc n
    plocs(n, 1) = x : plocs(n, 2) = y : plocs(n, 3) = -1
    box x, y, PWIDTH, PHEIGHT,, rgb(gray)
    inc x, PWIDTH+2*PLYGAP
  next i
end sub

' Assign segments to random partition boxes
sub AssignPartitionSegments
  local i, r, t
  for i = 1 to npart
    seglocindex(i) = i
  next i
  for i = 1 to npart
    r = RandInt(1, npart)
    t = seglocindex(r)
    seglocindex(r) = seglocindex(i)
    seglocindex(i) = t
  next i
end sub

' Draw the Partition Segments in Randomized Boxes around the grid
sub DrawPartitionSegments first
  local p, i, t, r, x, y, s
  for p = 1 to npart
    s = seglocindex(p)
    x = plocs(s, 1) : y = plocs(s, 2)
    box x, y, PWIDTH, PHEIGHT,, rgb(gray)
    plocs(s, 4) = p
    if first then
      plocs(s, 3) = 1
    end if
    if plocs(seglocindex(p), 3) <> 0 then
      DrawSegmentInBox p
    end if
  next p
end sub

' Make the segment Bounding Boxes
' They are used for creating a sprite for the active segment.
sub MakeSegBBs
  local row, h, i, n, p, x, y, xsum, ysum, sz
  local pxc, pyc, px1, py1, px2, py2
  local dx, dy, min_dx, min_dy, dxi, dyi
  sz = CSIZE
  local float ht = sqr(3)*sz - 1
  local float hs = 3.0*sz
  for p = 1 to npart
    seg_bb(p, 1) = mm.hres : seg_bb(p, 2) = mm.vres
    seg_bb(p, 3) = 0 : seg_bb(p, 4) = 0
  next p
  for row = 1 to ncrows(size)
    n = nchpr(size, row)
    y = -(ncrows(size)\2)*0.5*ht + (row-1)*0.5*ht
    for h = 1 to n
      x = -(n-1)*0.5*hs + (h-1)*hs
      p = pboard(row, h)
      if p = 0 then
        cls
        print "ERROR -- empty cell at (";row;",";h;")"
        end
      end if
      pxc = CX+x : px1 = pxc-WIDTH/2 : px2 = pxc+WIDTH/2+1
      pyc = CY+y : py1 = pyc-HEIGHT/2-1 : py2 = pyc+HEIGHT/2+2
      if px1 < seg_bb(p, 1) then seg_bb(p, 1) = px1
      if px2 > seg_bb(p, 3) then seg_bb(p, 3) = px2
      if py1 < seg_bb(p, 2) then seg_bb(p, 2) = py1
      if py2 > seg_bb(p, 4) then seg_bb(p, 4) = py2
    next h
  next row
end sub

' Compute the Centroids of each partition segment
' with respect to (0, 0). This is only used for positioning
' the segments inside the segment boxes.
sub MakeCentroids sz
  local row, h, i, n, p, x, y, xsum, ysum
  for p = 1 to npart
    xsum = 0 : ysum = 0
    for i = 1 to seg_nhexes(p)
      inc xsum, small_seg_pixel_coords(p, i, 1)
      inc ysum, small_seg_pixel_coords(p, i, 2)
    next i
    pcentroids(p, 1) = int((1.0*xsum)/(1.0*seg_nhexes(p)) + 0.5)
    pcentroids(p, 2) = int((1.0*ysum)/(1.0*seg_nhexes(p)) + 0.5) + 1
  next p
end sub

' Compute Indices for Segment hex closest to its cartesian center
sub GetSegCenters
  local p, i, crow, ccol, crowx, ccolx, sci
  local minrow, maxrow, mincol, maxcol, pwidth, pheight
  local float avrow, avcol, dr, dc, mindr, mindc
  local cmrow, cmh, smrow, smh, srow, sh
  for p = 1 to npart
    avrow = 0 : avcol = 0
    minrow = 100 : maxrow = 0 : mincol = 100 : maxcol = 0
    for i = 1 to seg_nhexes(p)
      crow = seg_hex_coords(p, i, 1)
      ccol = seg_hex_coords(p, i, 2)
      if crow < minrow then minrow = crow
      if crow > maxrow then maxrow = crow
      if ccol < mincol then mincol = ccol
      if ccol > maxcol then maxcol = ccol
      inc avrow, crow : inc avcol, ccol
    next i
    avrow = (1.0*avrow)/seg_nhexes(p)
    avcol = (1.0*avcol)/seg_nhexes(p)
    mindr = 100.0 : maxdr = 0
    mindc = 100.0 : maxdc = 0
    pwidth = maxcol - mincol
    pheight = maxrow - minrow
    for i = 1 to seg_nhexes(p)
      crow = seg_hex_coords(p, i, 1)
      ccol = seg_hex_coords(p, i, 2)
      dr = abs(avrow - crow)
      dc = abs(avcol - ccol)
      if dr < mindr then
        mindr = dr
        crowx = i    
      end if
      if dc < mindc then
        mindc = dc
        ccolx = i
      end if
    next i
    if pheight >= pwidth then
      seg_center_index(p) = crowx
    else
      seg_center_index(p) = ccolx
    end if
    sci = seg_center_index(p)
    cmrow = seg_hex_coords(p, sci, 1)
    cmh   = seg_hex_coords(p, sci, 2)
    C2S size, cmrow, cmh, smrow, smh
    for i = 1 to seg_nhexes(p)
      crow = seg_hex_coords(p, i, 1)
      ccol = seg_hex_coords(p, i, 2)
      C2S size, crow, ccol, srow, sh
      seg_diff_hex_coords(p, i, 1) = srow - smrow
      seg_diff_hex_coords(p, i, 2) = sh - smh
    next i
  next p
end sub

' Draw a minified partition segment in the specified box
sub DrawSegmentInBox p
  local x, y, rp
  MakeCentroids RSIZE
  rp = seglocindex(p)
  x = plocs(rp, 1) + PWIDTH\2 - pcentroids(p, 1)
  y = plocs(rp, 2) + PHEIGHT\2 - pcentroids(p, 2) - 7
  DrawGrid size, x, y, RSIZE, p, pboard(), LB_NONE
end sub

' Erase the partition segment in the specified box
sub EraseSegmentFromBox rp
  box plocs(rp, 1), plocs(rp, 2), PWIDTH, PHEIGHT,, rgb(grey), rgb(black)
end sub

' Return the box index for the specified segment
function GetSegmentBoxIndex(p)
  local i
  for i = 1 to npart
    if plocs(i, 4) = p then
      GetSegmentBoxIndex = i
      exit function
    end if
  next i
  GetSegmentBoxIndex = 0
end function

' Function to detect success at filling the grid
function IsComplete()
  local crow, ccol
  IsComplete = 1
  for crow = 1 to ncrows(size)
    for ccol = 1 to nchpr(size, crow)
      if board(crow, ccol) = 0 then
        IsComplete = 0
        exit function
      end if
    next ccol
  next crow
end function

' Quit the program
sub Quit
  settick 0, UpdateCursor
  controller mouse close
  cls
  end
end sub

' Show a hint
sub ShowHint
  local p, row, h, s
  p = 2
  ClearBoard
  LoadPartition
  for row = 1 to ncrows(size)
    for h = 1 to nchpr(size, row)
      if pboard(row, h) = p then board(row, h) = p
    next h
  next row
  s = seglocindex(p)
  plocs(s, 3) = 0
  DrawScreen 
end sub

' Clear all segments from the board
sub ClearBoard
  local row, h, rp, s
  for row = 1 to ncrows(size)
    for h = 1 to nchpr(size, row)
      board(row, h) = 0
    next h
  next row
  for rp = 1 to MAX_NPARTS
    if plocs(rp, 4) > 0 then plocs(rp, 3) = 1
  next rp
end sub

' Show victory
sub ShowComplete
  won = 1
  DrawBanner "COMPLETE!", 5, rgb(blue), rgb(gold)
  DrawBoard
  SavePartition not reset_solved
end sub

' Draw a Banner at screen top
' b$: banner text
' f: font number
' fc: font color
sub DrawBanner b$, f, fc, bc
  if bc <> 0 then
    box mm.hres\2-BANN_W\2, BANN_Y, BANN_W, BANN_H,, bc, bc
  end if
  text mm.hres\2, BANN_Y+BANN_H\2, b$, "CM", f,, fc, -1
end sub

' Help
sub ShowHelp
  local z$ = INKEY$
  cls
  text mm.hres\2, 0, "Directions for Playing HexParty", "CT", 4,, rgb(green)
  print @(0, 20) ""
  print "HexParty is a puzzle where you are presented with a board made up of hexagons,"
  print "and a set of partitions that somehow have to be fitted into the board so that"
  print "all cells are filled. This also implies that none of the partitions overlap or"
  print "extend outside the board edges."
  print ""
  print "When the program is started, you are presented with a screen that lists all existing"
  print "puzzles, sorted by grid size and by puzzle number. Use the UP, DOWN, LEFT, and RIGHT"
  print "keyboard keys to choose a puzzle, then press ENTER to begin working it."
  print ""
  print "The board is then shown in the center of the screen, with minified versions of the"
  print "partitions arrayed around the board in boxes. The locations of the segments in the
  print "boxes have nothing to do with their correct locations in the main grid.
  print ""
  print "To place a segment onto the board, click the segment box for the segment you want to"
  print "place. The full-size segment will appear and will be dragged around by the mouse."
  print "Move it to the location you think it belongs in the main grid and left-click again"
  print "to drop it onto the grid. If the segment overlaps any existing segment that is"
  print "already on the grid, or if the segment extends outside the grid, you will be unable"
  print "to drop it, and a Beep will sound. To quit dragging the segment if it cannot be placed,
  print "drag it outside the main grid and click the left mouse key, or press the DELETE or"
  print "BACKSPACE keyboard key.
  print ""
  print "To adjust the position of a placed segment, position the mouse over one of its parts"
  print "and left-click the mouse. The segment will become movable again."
  print ""
  print "Move all segments onto the board, and when you get them all correctly arranged, you"
  print "will be rewarded with a big 'COMPLETE!' sign at the top of the screen."
  print ""
  print "If you get stuck, you can press the 'H' key to get a computer-chosen segment placed"
  print "correctly. However, this also erases all your previously placed segments. 
  print ""
  print "To play a new puzzle, press the HOME key. To quit press the ESCAPE key. If the supplied"
  print "puzzles are not enough, use the 'MakePartition.bas' program to make your own puzzles."
  print ""
  print "As you solve puzzles, they will be marked as solved in the puzzle choice screen. This"
  print "does not keep you from working on them again."
  text mm.hres\2, 589, "Press Any Key to Continue", "CB"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' Cartesian Hex data:
' Each row begins with number of rows in board,
' followed by number of hexes per row.
data  7, 2, 1, 2, 3, 2, 1, 2
data 13, 3, 2, 3, 4, 3, 4, 5, 4, 3, 4, 3, 2, 3
data 19, 4, 3, 4, 5, 4, 5, 6, 5, 6, 7, 6, 5, 6, 5, 4, 5, 4, 3, 4

' Max number of partition segments per grid size
data 6, 18, 13

' Grid center coords (cartesian)
data 4, 2, 7, 3, 10, 4

' Color Pallette
data rgb(red), rgb(green), rgb(blue), rgb(yellow), rgb(cyan), rgb(magenta)
data rgb(179, 18, 84), rgb(128, 65, 0), rgb(255, 142, 0), rgb(255, 178, 178)
data rgb(38, 128, 49), rgb(127, 127, 255), rgb(142, 25, 255)



